﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>
#include <aws/iot/model/TopicRuleDestinationStatus.h>

#include <utility>

namespace Aws {
namespace IoT {
namespace Model {

/**
 */
class UpdateTopicRuleDestinationRequest : public IoTRequest {
 public:
  AWS_IOT_API UpdateTopicRuleDestinationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateTopicRuleDestination"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ARN of the topic rule destination.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  UpdateTopicRuleDestinationRequest& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the topic rule destination. Valid values are:</p> <dl>
   * <dt>IN_PROGRESS</dt> <dd> <p>A topic rule destination was created but has not
   * been confirmed. You can set <code>status</code> to <code>IN_PROGRESS</code> by
   * calling <code>UpdateTopicRuleDestination</code>. Calling
   * <code>UpdateTopicRuleDestination</code> causes a new confirmation challenge to
   * be sent to your confirmation endpoint.</p> </dd> <dt>ENABLED</dt> <dd>
   * <p>Confirmation was completed, and traffic to this destination is allowed. You
   * can set <code>status</code> to <code>DISABLED</code> by calling
   * <code>UpdateTopicRuleDestination</code>.</p> </dd> <dt>DISABLED</dt> <dd>
   * <p>Confirmation was completed, and traffic to this destination is not allowed.
   * You can set <code>status</code> to <code>ENABLED</code> by calling
   * <code>UpdateTopicRuleDestination</code>.</p> </dd> <dt>ERROR</dt> <dd>
   * <p>Confirmation could not be completed, for example if the confirmation timed
   * out. You can call <code>GetTopicRuleDestination</code> for details about the
   * error. You can set <code>status</code> to <code>IN_PROGRESS</code> by calling
   * <code>UpdateTopicRuleDestination</code>. Calling
   * <code>UpdateTopicRuleDestination</code> causes a new confirmation challenge to
   * be sent to your confirmation endpoint.</p> </dd> </dl>
   */
  inline TopicRuleDestinationStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(TopicRuleDestinationStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline UpdateTopicRuleDestinationRequest& WithStatus(TopicRuleDestinationStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  TopicRuleDestinationStatus m_status{TopicRuleDestinationStatus::NOT_SET};
  bool m_statusHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
