﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/storagegateway/StorageGatewayRequest.h>
#include <aws/storagegateway/StorageGateway_EXPORTS.h>
#include <aws/storagegateway/model/Tag.h>

#include <utility>

namespace Aws {
namespace StorageGateway {
namespace Model {

/**
 * <p>A JSON object containing one or more of the following fields:</p> <ul> <li>
 * <p> <a>CreateStorediSCSIVolumeInput$DiskId</a> </p> </li> <li> <p>
 * <a>CreateStorediSCSIVolumeInput$NetworkInterfaceId</a> </p> </li> <li> <p>
 * <a>CreateStorediSCSIVolumeInput$PreserveExistingData</a> </p> </li> <li> <p>
 * <a>CreateStorediSCSIVolumeInput$SnapshotId</a> </p> </li> <li> <p>
 * <a>CreateStorediSCSIVolumeInput$TargetName</a> </p> </li> </ul><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/storagegateway-2013-06-30/CreateStorediSCSIVolumeInput">AWS
 * API Reference</a></p>
 */
class CreateStorediSCSIVolumeRequest : public StorageGatewayRequest {
 public:
  AWS_STORAGEGATEWAY_API CreateStorediSCSIVolumeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateStorediSCSIVolume"; }

  AWS_STORAGEGATEWAY_API Aws::String SerializePayload() const override;

  AWS_STORAGEGATEWAY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{

  inline const Aws::String& GetGatewayARN() const { return m_gatewayARN; }
  inline bool GatewayARNHasBeenSet() const { return m_gatewayARNHasBeenSet; }
  template <typename GatewayARNT = Aws::String>
  void SetGatewayARN(GatewayARNT&& value) {
    m_gatewayARNHasBeenSet = true;
    m_gatewayARN = std::forward<GatewayARNT>(value);
  }
  template <typename GatewayARNT = Aws::String>
  CreateStorediSCSIVolumeRequest& WithGatewayARN(GatewayARNT&& value) {
    SetGatewayARN(std::forward<GatewayARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier for the gateway local disk that is configured as a
   * stored volume. Use <a
   * href="https://docs.aws.amazon.com/storagegateway/latest/userguide/API_ListLocalDisks.html">ListLocalDisks</a>
   * to list disk IDs for a gateway.</p>
   */
  inline const Aws::String& GetDiskId() const { return m_diskId; }
  inline bool DiskIdHasBeenSet() const { return m_diskIdHasBeenSet; }
  template <typename DiskIdT = Aws::String>
  void SetDiskId(DiskIdT&& value) {
    m_diskIdHasBeenSet = true;
    m_diskId = std::forward<DiskIdT>(value);
  }
  template <typename DiskIdT = Aws::String>
  CreateStorediSCSIVolumeRequest& WithDiskId(DiskIdT&& value) {
    SetDiskId(std::forward<DiskIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The snapshot ID (e.g., "snap-1122aabb") of the snapshot to restore as the new
   * stored volume. Specify this field if you want to create the iSCSI storage volume
   * from a snapshot; otherwise, do not include this field. To list snapshots for
   * your account use <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/ApiReference-query-DescribeSnapshots.html">DescribeSnapshots</a>
   * in the <i>Amazon Elastic Compute Cloud API Reference</i>.</p>
   */
  inline const Aws::String& GetSnapshotId() const { return m_snapshotId; }
  inline bool SnapshotIdHasBeenSet() const { return m_snapshotIdHasBeenSet; }
  template <typename SnapshotIdT = Aws::String>
  void SetSnapshotId(SnapshotIdT&& value) {
    m_snapshotIdHasBeenSet = true;
    m_snapshotId = std::forward<SnapshotIdT>(value);
  }
  template <typename SnapshotIdT = Aws::String>
  CreateStorediSCSIVolumeRequest& WithSnapshotId(SnapshotIdT&& value) {
    SetSnapshotId(std::forward<SnapshotIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set to <code>true</code> if you want to preserve the data on the local disk.
   * Otherwise, set to <code>false</code> to create an empty volume.</p> <p>Valid
   * Values: <code>true</code> | <code>false</code> </p>
   */
  inline bool GetPreserveExistingData() const { return m_preserveExistingData; }
  inline bool PreserveExistingDataHasBeenSet() const { return m_preserveExistingDataHasBeenSet; }
  inline void SetPreserveExistingData(bool value) {
    m_preserveExistingDataHasBeenSet = true;
    m_preserveExistingData = value;
  }
  inline CreateStorediSCSIVolumeRequest& WithPreserveExistingData(bool value) {
    SetPreserveExistingData(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the iSCSI target used by an initiator to connect to a volume and
   * used as a suffix for the target ARN. For example, specifying
   * <code>TargetName</code> as <i>myvolume</i> results in the target ARN of
   * <code>arn:aws:storagegateway:us-east-2:111122223333:gateway/sgw-12A3456B/target/iqn.1997-05.com.amazon:myvolume</code>.
   * The target name must be unique across all volumes on a gateway.</p> <p>If you
   * don't specify a value, Storage Gateway uses the value that was previously used
   * for this volume as the new target name.</p>
   */
  inline const Aws::String& GetTargetName() const { return m_targetName; }
  inline bool TargetNameHasBeenSet() const { return m_targetNameHasBeenSet; }
  template <typename TargetNameT = Aws::String>
  void SetTargetName(TargetNameT&& value) {
    m_targetNameHasBeenSet = true;
    m_targetName = std::forward<TargetNameT>(value);
  }
  template <typename TargetNameT = Aws::String>
  CreateStorediSCSIVolumeRequest& WithTargetName(TargetNameT&& value) {
    SetTargetName(std::forward<TargetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The network interface of the gateway on which to expose the iSCSI target.
   * Accepts IPv4 and IPv6 addresses. Use <a>DescribeGatewayInformation</a> to get a
   * list of the network interfaces available on a gateway.</p> <p>Valid Values: A
   * valid IP address.</p>
   */
  inline const Aws::String& GetNetworkInterfaceId() const { return m_networkInterfaceId; }
  inline bool NetworkInterfaceIdHasBeenSet() const { return m_networkInterfaceIdHasBeenSet; }
  template <typename NetworkInterfaceIdT = Aws::String>
  void SetNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    m_networkInterfaceIdHasBeenSet = true;
    m_networkInterfaceId = std::forward<NetworkInterfaceIdT>(value);
  }
  template <typename NetworkInterfaceIdT = Aws::String>
  CreateStorediSCSIVolumeRequest& WithNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    SetNetworkInterfaceId(std::forward<NetworkInterfaceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set to <code>true</code> to use Amazon S3 server-side encryption with your
   * own KMS key, or <code>false</code> to use a key managed by Amazon S3.
   * Optional.</p> <p>Valid Values: <code>true</code> | <code>false</code> </p>
   */
  inline bool GetKMSEncrypted() const { return m_kMSEncrypted; }
  inline bool KMSEncryptedHasBeenSet() const { return m_kMSEncryptedHasBeenSet; }
  inline void SetKMSEncrypted(bool value) {
    m_kMSEncryptedHasBeenSet = true;
    m_kMSEncrypted = value;
  }
  inline CreateStorediSCSIVolumeRequest& WithKMSEncrypted(bool value) {
    SetKMSEncrypted(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of a symmetric customer master key (CMK) used
   * for Amazon S3 server-side encryption. Storage Gateway does not support
   * asymmetric CMKs. This value can only be set when <code>KMSEncrypted</code> is
   * <code>true</code>. Optional.</p>
   */
  inline const Aws::String& GetKMSKey() const { return m_kMSKey; }
  inline bool KMSKeyHasBeenSet() const { return m_kMSKeyHasBeenSet; }
  template <typename KMSKeyT = Aws::String>
  void SetKMSKey(KMSKeyT&& value) {
    m_kMSKeyHasBeenSet = true;
    m_kMSKey = std::forward<KMSKeyT>(value);
  }
  template <typename KMSKeyT = Aws::String>
  CreateStorediSCSIVolumeRequest& WithKMSKey(KMSKeyT&& value) {
    SetKMSKey(std::forward<KMSKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of up to 50 tags that can be assigned to a stored volume. Each tag is
   * a key-value pair.</p>  <p>Valid characters for key and value are letters,
   * spaces, and numbers representable in UTF-8 format, and the following special
   * characters: + - = . _ : / @. The maximum length of a tag's key is 128
   * characters, and the maximum length for a tag's value is 256.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateStorediSCSIVolumeRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateStorediSCSIVolumeRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_gatewayARN;

  Aws::String m_diskId;

  Aws::String m_snapshotId;

  bool m_preserveExistingData{false};

  Aws::String m_targetName;

  Aws::String m_networkInterfaceId;

  bool m_kMSEncrypted{false};

  Aws::String m_kMSKey;

  Aws::Vector<Tag> m_tags;
  bool m_gatewayARNHasBeenSet = false;
  bool m_diskIdHasBeenSet = false;
  bool m_snapshotIdHasBeenSet = false;
  bool m_preserveExistingDataHasBeenSet = false;
  bool m_targetNameHasBeenSet = false;
  bool m_networkInterfaceIdHasBeenSet = false;
  bool m_kMSEncryptedHasBeenSet = false;
  bool m_kMSKeyHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace StorageGateway
}  // namespace Aws
